// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { filterInPlace } from "./_utils.ts";
export function deepMerge(record, other, options) {
  return deepMergeInternal(record, other, new Set(), options);
}
function deepMergeInternal(record, other, seen, options) {
  const result = {};
  const keys = new Set([
    ...getKeys(record),
    ...getKeys(other)
  ]);
  // Iterate through each key of other object and use correct merging strategy
  for (const key of keys){
    // Skip to prevent Object.prototype.__proto__ accessor property calls on non-Deno platforms
    if (key === "__proto__") {
      continue;
    }
    const a = record[key];
    if (!Object.hasOwn(other, key)) {
      result[key] = a;
      continue;
    }
    const b = other[key];
    if (isNonNullObject(a) && isNonNullObject(b) && !seen.has(a) && !seen.has(b)) {
      seen.add(a);
      seen.add(b);
      result[key] = mergeObjects(a, b, seen, options);
      continue;
    }
    // Override value
    result[key] = b;
  }
  return result;
}
function mergeObjects(left, right, seen, options = {
  arrays: "merge",
  sets: "merge",
  maps: "merge"
}) {
  // Recursively merge mergeable objects
  if (isMergeable(left) && isMergeable(right)) {
    return deepMergeInternal(left, right, seen, options);
  }
  if (isIterable(left) && isIterable(right)) {
    // Handle arrays
    if (Array.isArray(left) && Array.isArray(right)) {
      if (options.arrays === "merge") {
        return left.concat(right);
      }
      return right;
    }
    // Handle maps
    if (left instanceof Map && right instanceof Map) {
      if (options.maps === "merge") {
        return new Map([
          ...left,
          ...right
        ]);
      }
      return right;
    }
    // Handle sets
    if (left instanceof Set && right instanceof Set) {
      if (options.sets === "merge") {
        return new Set([
          ...left,
          ...right
        ]);
      }
      return right;
    }
  }
  return right;
}
/**
 * Test whether a value is mergeable or not
 * Builtins that look like objects, null and user defined classes
 * are not considered mergeable (it means that reference will be copied)
 */ function isMergeable(value) {
  return Object.getPrototypeOf(value) === Object.prototype;
}
function isIterable(value) {
  return typeof value[Symbol.iterator] === "function";
}
function isNonNullObject(value) {
  return value !== null && typeof value === "object";
}
function getKeys(record) {
  const result = Object.getOwnPropertySymbols(record);
  filterInPlace(result, (key)=>Object.prototype.propertyIsEnumerable.call(record, key));
  result.push(...Object.keys(record));
  return result;
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=5729930655758230209,2265190145882392849