// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/*!
 * Adapted directly from negotiator at https://github.com/jshttp/negotiator/
 * which is licensed as follows:
 *
 * (The MIT License)
 *
 * Copyright (c) 2012-2014 Federico Romero
 * Copyright (c) 2012-2014 Isaac Z. Schlueter
 * Copyright (c) 2014-2015 Douglas Christopher Wilson
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * 'Software'), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */ import { compareSpecs, isQuality } from "./common.ts";
const simpleMediaTypeRegExp = /^\s*([^\s\/;]+)\/([^;\s]+)\s*(?:;(.*))?$/;
function splitKeyValuePair(str) {
  const [key, value] = str.split("=");
  return [
    key.toLowerCase(),
    value
  ];
}
function parseMediaType(str, i) {
  const match = simpleMediaTypeRegExp.exec(str);
  if (!match) {
    return;
  }
  const [, type, subtype, parameters] = match;
  if (!type || !subtype) {
    return;
  }
  const params = Object.create(null);
  let q = 1;
  if (parameters) {
    const kvps = parameters.split(";").map((p)=>p.trim()).map(splitKeyValuePair);
    for (const [key, val] of kvps){
      const value = val && val[0] === `"` && val[val.length - 1] === `"` ? val.slice(1, val.length - 1) : val;
      if (key === "q" && value) {
        q = parseFloat(value);
        break;
      }
      params[key] = value;
    }
  }
  return {
    type,
    subtype,
    params,
    q,
    i
  };
}
function parseAccept(accept) {
  const accepts = accept.split(",").map((p)=>p.trim());
  const mediaTypes = [];
  for (const [index, accept] of accepts.entries()){
    const mediaType = parseMediaType(accept.trim(), index);
    if (mediaType) {
      mediaTypes.push(mediaType);
    }
  }
  return mediaTypes;
}
function getFullType(spec) {
  return `${spec.type}/${spec.subtype}`;
}
function specify(type, spec, index) {
  const p = parseMediaType(type, index);
  if (!p) {
    return;
  }
  let s = 0;
  if (spec.type.toLowerCase() === p.type.toLowerCase()) {
    s |= 4;
  } else if (spec.type !== "*") {
    return;
  }
  if (spec.subtype.toLowerCase() === p.subtype.toLowerCase()) {
    s |= 2;
  } else if (spec.subtype !== "*") {
    return;
  }
  const keys = Object.keys(spec.params);
  if (keys.length) {
    if (keys.every((key)=>(spec.params[key] || "").toLowerCase() === (p.params[key] || "").toLowerCase())) {
      s |= 1;
    } else {
      return;
    }
  }
  return {
    i: index,
    o: spec.o,
    q: spec.q,
    s
  };
}
function getMediaTypePriority(type, accepted, index) {
  let priority = {
    o: -1,
    q: 0,
    s: 0,
    i: index
  };
  for (const accepts of accepted){
    const spec = specify(type, accepts, index);
    if (spec && ((priority.s || 0) - (spec.s || 0) || (priority.q || 0) - (spec.q || 0) || (priority.o || 0) - (spec.o || 0)) < 0) {
      priority = spec;
    }
  }
  return priority;
}
export function preferredMediaTypes(accept, provided) {
  const accepts = parseAccept(accept === undefined ? "*/*" : accept || "");
  if (!provided) {
    return accepts.filter(isQuality).sort(compareSpecs).map(getFullType);
  }
  const priorities = provided.map((type, index)=>{
    return getMediaTypePriority(type, accepts, index);
  });
  return priorities.filter(isQuality).sort(compareSpecs).map((priority)=>provided[priorities.indexOf(priority)]);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vanNyLmlvL0BzdGQvaHR0cC8wLjIyNC41L19uZWdvdGlhdGlvbi9tZWRpYV90eXBlLnRzIl0sInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAyMDE4LTIwMjQgdGhlIERlbm8gYXV0aG9ycy4gQWxsIHJpZ2h0cyByZXNlcnZlZC4gTUlUIGxpY2Vuc2UuXG4vKiFcbiAqIEFkYXB0ZWQgZGlyZWN0bHkgZnJvbSBuZWdvdGlhdG9yIGF0IGh0dHBzOi8vZ2l0aHViLmNvbS9qc2h0dHAvbmVnb3RpYXRvci9cbiAqIHdoaWNoIGlzIGxpY2Vuc2VkIGFzIGZvbGxvd3M6XG4gKlxuICogKFRoZSBNSVQgTGljZW5zZSlcbiAqXG4gKiBDb3B5cmlnaHQgKGMpIDIwMTItMjAxNCBGZWRlcmljbyBSb21lcm9cbiAqIENvcHlyaWdodCAoYykgMjAxMi0yMDE0IElzYWFjIFouIFNjaGx1ZXRlclxuICogQ29weXJpZ2h0IChjKSAyMDE0LTIwMTUgRG91Z2xhcyBDaHJpc3RvcGhlciBXaWxzb25cbiAqXG4gKiBQZXJtaXNzaW9uIGlzIGhlcmVieSBncmFudGVkLCBmcmVlIG9mIGNoYXJnZSwgdG8gYW55IHBlcnNvbiBvYnRhaW5pbmdcbiAqIGEgY29weSBvZiB0aGlzIHNvZnR3YXJlIGFuZCBhc3NvY2lhdGVkIGRvY3VtZW50YXRpb24gZmlsZXMgKHRoZVxuICogJ1NvZnR3YXJlJyksIHRvIGRlYWwgaW4gdGhlIFNvZnR3YXJlIHdpdGhvdXQgcmVzdHJpY3Rpb24sIGluY2x1ZGluZ1xuICogd2l0aG91dCBsaW1pdGF0aW9uIHRoZSByaWdodHMgdG8gdXNlLCBjb3B5LCBtb2RpZnksIG1lcmdlLCBwdWJsaXNoLFxuICogZGlzdHJpYnV0ZSwgc3VibGljZW5zZSwgYW5kL29yIHNlbGwgY29waWVzIG9mIHRoZSBTb2Z0d2FyZSwgYW5kIHRvXG4gKiBwZXJtaXQgcGVyc29ucyB0byB3aG9tIHRoZSBTb2Z0d2FyZSBpcyBmdXJuaXNoZWQgdG8gZG8gc28sIHN1YmplY3QgdG9cbiAqIHRoZSBmb2xsb3dpbmcgY29uZGl0aW9uczpcbiAqXG4gKiBUaGUgYWJvdmUgY29weXJpZ2h0IG5vdGljZSBhbmQgdGhpcyBwZXJtaXNzaW9uIG5vdGljZSBzaGFsbCBiZVxuICogaW5jbHVkZWQgaW4gYWxsIGNvcGllcyBvciBzdWJzdGFudGlhbCBwb3J0aW9ucyBvZiB0aGUgU29mdHdhcmUuXG4gKlxuICogVEhFIFNPRlRXQVJFIElTIFBST1ZJREVEICdBUyBJUycsIFdJVEhPVVQgV0FSUkFOVFkgT0YgQU5ZIEtJTkQsXG4gKiBFWFBSRVNTIE9SIElNUExJRUQsIElOQ0xVRElORyBCVVQgTk9UIExJTUlURUQgVE8gVEhFIFdBUlJBTlRJRVMgT0ZcbiAqIE1FUkNIQU5UQUJJTElUWSwgRklUTkVTUyBGT1IgQSBQQVJUSUNVTEFSIFBVUlBPU0UgQU5EIE5PTklORlJJTkdFTUVOVC5cbiAqIElOIE5PIEVWRU5UIFNIQUxMIFRIRSBBVVRIT1JTIE9SIENPUFlSSUdIVCBIT0xERVJTIEJFIExJQUJMRSBGT1IgQU5ZXG4gKiBDTEFJTSwgREFNQUdFUyBPUiBPVEhFUiBMSUFCSUxJVFksIFdIRVRIRVIgSU4gQU4gQUNUSU9OIE9GIENPTlRSQUNULFxuICogVE9SVCBPUiBPVEhFUldJU0UsIEFSSVNJTkcgRlJPTSwgT1VUIE9GIE9SIElOIENPTk5FQ1RJT04gV0lUSCBUSEVcbiAqIFNPRlRXQVJFIE9SIFRIRSBVU0UgT1IgT1RIRVIgREVBTElOR1MgSU4gVEhFIFNPRlRXQVJFLlxuICovXG5cbmltcG9ydCB7IGNvbXBhcmVTcGVjcywgaXNRdWFsaXR5LCB0eXBlIFNwZWNpZmljaXR5IH0gZnJvbSBcIi4vY29tbW9uLnRzXCI7XG5cbmludGVyZmFjZSBNZWRpYVR5cGVTcGVjaWZpY2l0eSBleHRlbmRzIFNwZWNpZmljaXR5IHtcbiAgdHlwZTogc3RyaW5nO1xuICBzdWJ0eXBlOiBzdHJpbmc7XG4gIHBhcmFtczogeyBbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHVuZGVmaW5lZCB9O1xufVxuXG5jb25zdCBzaW1wbGVNZWRpYVR5cGVSZWdFeHAgPSAvXlxccyooW15cXHNcXC87XSspXFwvKFteO1xcc10rKVxccyooPzo7KC4qKSk/JC87XG5cbmZ1bmN0aW9uIHNwbGl0S2V5VmFsdWVQYWlyKHN0cjogc3RyaW5nKTogW3N0cmluZywgc3RyaW5nIHwgdW5kZWZpbmVkXSB7XG4gIGNvbnN0IFtrZXksIHZhbHVlXSA9IHN0ci5zcGxpdChcIj1cIik7XG4gIHJldHVybiBba2V5IS50b0xvd2VyQ2FzZSgpLCB2YWx1ZV07XG59XG5cbmZ1bmN0aW9uIHBhcnNlTWVkaWFUeXBlKFxuICBzdHI6IHN0cmluZyxcbiAgaTogbnVtYmVyLFxuKTogTWVkaWFUeXBlU3BlY2lmaWNpdHkgfCB1bmRlZmluZWQge1xuICBjb25zdCBtYXRjaCA9IHNpbXBsZU1lZGlhVHlwZVJlZ0V4cC5leGVjKHN0cik7XG5cbiAgaWYgKCFtYXRjaCkge1xuICAgIHJldHVybjtcbiAgfVxuXG4gIGNvbnN0IFssIHR5cGUsIHN1YnR5cGUsIHBhcmFtZXRlcnNdID0gbWF0Y2g7XG4gIGlmICghdHlwZSB8fCAhc3VidHlwZSkge1xuICAgIHJldHVybjtcbiAgfVxuXG4gIGNvbnN0IHBhcmFtczogeyBbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHVuZGVmaW5lZCB9ID0gT2JqZWN0LmNyZWF0ZShudWxsKTtcbiAgbGV0IHEgPSAxO1xuICBpZiAocGFyYW1ldGVycykge1xuICAgIGNvbnN0IGt2cHMgPSBwYXJhbWV0ZXJzLnNwbGl0KFwiO1wiKS5tYXAoKHApID0+IHAudHJpbSgpKS5tYXAoXG4gICAgICBzcGxpdEtleVZhbHVlUGFpcixcbiAgICApO1xuXG4gICAgZm9yIChjb25zdCBba2V5LCB2YWxdIG9mIGt2cHMpIHtcbiAgICAgIGNvbnN0IHZhbHVlID0gdmFsICYmIHZhbFswXSA9PT0gYFwiYCAmJiB2YWxbdmFsLmxlbmd0aCAtIDFdID09PSBgXCJgXG4gICAgICAgID8gdmFsLnNsaWNlKDEsIHZhbC5sZW5ndGggLSAxKVxuICAgICAgICA6IHZhbDtcblxuICAgICAgaWYgKGtleSA9PT0gXCJxXCIgJiYgdmFsdWUpIHtcbiAgICAgICAgcSA9IHBhcnNlRmxvYXQodmFsdWUpO1xuICAgICAgICBicmVhaztcbiAgICAgIH1cblxuICAgICAgcGFyYW1zW2tleV0gPSB2YWx1ZTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4geyB0eXBlLCBzdWJ0eXBlLCBwYXJhbXMsIHEsIGkgfTtcbn1cblxuZnVuY3Rpb24gcGFyc2VBY2NlcHQoYWNjZXB0OiBzdHJpbmcpOiBNZWRpYVR5cGVTcGVjaWZpY2l0eVtdIHtcbiAgY29uc3QgYWNjZXB0cyA9IGFjY2VwdC5zcGxpdChcIixcIikubWFwKChwKSA9PiBwLnRyaW0oKSk7XG5cbiAgY29uc3QgbWVkaWFUeXBlczogTWVkaWFUeXBlU3BlY2lmaWNpdHlbXSA9IFtdO1xuICBmb3IgKGNvbnN0IFtpbmRleCwgYWNjZXB0XSBvZiBhY2NlcHRzLmVudHJpZXMoKSkge1xuICAgIGNvbnN0IG1lZGlhVHlwZSA9IHBhcnNlTWVkaWFUeXBlKGFjY2VwdC50cmltKCksIGluZGV4KTtcblxuICAgIGlmIChtZWRpYVR5cGUpIHtcbiAgICAgIG1lZGlhVHlwZXMucHVzaChtZWRpYVR5cGUpO1xuICAgIH1cbiAgfVxuXG4gIHJldHVybiBtZWRpYVR5cGVzO1xufVxuXG5mdW5jdGlvbiBnZXRGdWxsVHlwZShzcGVjOiBNZWRpYVR5cGVTcGVjaWZpY2l0eSkge1xuICByZXR1cm4gYCR7c3BlYy50eXBlfS8ke3NwZWMuc3VidHlwZX1gO1xufVxuXG5mdW5jdGlvbiBzcGVjaWZ5KFxuICB0eXBlOiBzdHJpbmcsXG4gIHNwZWM6IE1lZGlhVHlwZVNwZWNpZmljaXR5LFxuICBpbmRleDogbnVtYmVyLFxuKTogU3BlY2lmaWNpdHkgfCB1bmRlZmluZWQge1xuICBjb25zdCBwID0gcGFyc2VNZWRpYVR5cGUodHlwZSwgaW5kZXgpO1xuXG4gIGlmICghcCkge1xuICAgIHJldHVybjtcbiAgfVxuXG4gIGxldCBzID0gMDtcblxuICBpZiAoc3BlYy50eXBlLnRvTG93ZXJDYXNlKCkgPT09IHAudHlwZS50b0xvd2VyQ2FzZSgpKSB7XG4gICAgcyB8PSA0O1xuICB9IGVsc2UgaWYgKHNwZWMudHlwZSAhPT0gXCIqXCIpIHtcbiAgICByZXR1cm47XG4gIH1cblxuICBpZiAoc3BlYy5zdWJ0eXBlLnRvTG93ZXJDYXNlKCkgPT09IHAuc3VidHlwZS50b0xvd2VyQ2FzZSgpKSB7XG4gICAgcyB8PSAyO1xuICB9IGVsc2UgaWYgKHNwZWMuc3VidHlwZSAhPT0gXCIqXCIpIHtcbiAgICByZXR1cm47XG4gIH1cblxuICBjb25zdCBrZXlzID0gT2JqZWN0LmtleXMoc3BlYy5wYXJhbXMpO1xuICBpZiAoa2V5cy5sZW5ndGgpIHtcbiAgICBpZiAoXG4gICAgICBrZXlzLmV2ZXJ5KChrZXkpID0+XG4gICAgICAgIChzcGVjLnBhcmFtc1trZXldIHx8IFwiXCIpLnRvTG93ZXJDYXNlKCkgPT09XG4gICAgICAgICAgKHAucGFyYW1zW2tleV0gfHwgXCJcIikudG9Mb3dlckNhc2UoKVxuICAgICAgKVxuICAgICkge1xuICAgICAgcyB8PSAxO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIHtcbiAgICBpOiBpbmRleCxcbiAgICBvOiBzcGVjLm8sXG4gICAgcTogc3BlYy5xLFxuICAgIHMsXG4gIH07XG59XG5cbmZ1bmN0aW9uIGdldE1lZGlhVHlwZVByaW9yaXR5KFxuICB0eXBlOiBzdHJpbmcsXG4gIGFjY2VwdGVkOiBNZWRpYVR5cGVTcGVjaWZpY2l0eVtdLFxuICBpbmRleDogbnVtYmVyLFxuKSB7XG4gIGxldCBwcmlvcml0eTogU3BlY2lmaWNpdHkgPSB7IG86IC0xLCBxOiAwLCBzOiAwLCBpOiBpbmRleCB9O1xuXG4gIGZvciAoY29uc3QgYWNjZXB0cyBvZiBhY2NlcHRlZCkge1xuICAgIGNvbnN0IHNwZWMgPSBzcGVjaWZ5KHR5cGUsIGFjY2VwdHMsIGluZGV4KTtcblxuICAgIGlmIChcbiAgICAgIHNwZWMgJiZcbiAgICAgICgocHJpb3JpdHkucyB8fCAwKSAtIChzcGVjLnMgfHwgMCkgfHxcbiAgICAgICAgICAocHJpb3JpdHkucSB8fCAwKSAtIChzcGVjLnEgfHwgMCkgfHxcbiAgICAgICAgICAocHJpb3JpdHkubyB8fCAwKSAtIChzcGVjLm8gfHwgMCkpIDwgMFxuICAgICkge1xuICAgICAgcHJpb3JpdHkgPSBzcGVjO1xuICAgIH1cbiAgfVxuXG4gIHJldHVybiBwcmlvcml0eTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHByZWZlcnJlZE1lZGlhVHlwZXMoXG4gIGFjY2VwdD86IHN0cmluZyB8IG51bGwsXG4gIHByb3ZpZGVkPzogc3RyaW5nW10sXG4pOiBzdHJpbmdbXSB7XG4gIGNvbnN0IGFjY2VwdHMgPSBwYXJzZUFjY2VwdChhY2NlcHQgPT09IHVuZGVmaW5lZCA/IFwiKi8qXCIgOiBhY2NlcHQgfHwgXCJcIik7XG5cbiAgaWYgKCFwcm92aWRlZCkge1xuICAgIHJldHVybiBhY2NlcHRzXG4gICAgICAuZmlsdGVyKGlzUXVhbGl0eSlcbiAgICAgIC5zb3J0KGNvbXBhcmVTcGVjcylcbiAgICAgIC5tYXAoZ2V0RnVsbFR5cGUpO1xuICB9XG5cbiAgY29uc3QgcHJpb3JpdGllcyA9IHByb3ZpZGVkLm1hcCgodHlwZSwgaW5kZXgpID0+IHtcbiAgICByZXR1cm4gZ2V0TWVkaWFUeXBlUHJpb3JpdHkodHlwZSwgYWNjZXB0cywgaW5kZXgpO1xuICB9KTtcblxuICByZXR1cm4gcHJpb3JpdGllc1xuICAgIC5maWx0ZXIoaXNRdWFsaXR5KVxuICAgIC5zb3J0KGNvbXBhcmVTcGVjcylcbiAgICAubWFwKChwcmlvcml0eSkgPT4gcHJvdmlkZWRbcHJpb3JpdGllcy5pbmRleE9mKHByaW9yaXR5KV0hKTtcbn1cbiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSwwRUFBMEU7QUFDMUU7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Q0E0QkMsR0FFRCxTQUFTLFlBQVksRUFBRSxTQUFTLFFBQTBCLGNBQWM7QUFReEUsTUFBTSx3QkFBd0I7QUFFOUIsU0FBUyxrQkFBa0IsR0FBVztFQUNwQyxNQUFNLENBQUMsS0FBSyxNQUFNLEdBQUcsSUFBSSxLQUFLLENBQUM7RUFDL0IsT0FBTztJQUFDLElBQUssV0FBVztJQUFJO0dBQU07QUFDcEM7QUFFQSxTQUFTLGVBQ1AsR0FBVyxFQUNYLENBQVM7RUFFVCxNQUFNLFFBQVEsc0JBQXNCLElBQUksQ0FBQztFQUV6QyxJQUFJLENBQUMsT0FBTztJQUNWO0VBQ0Y7RUFFQSxNQUFNLEdBQUcsTUFBTSxTQUFTLFdBQVcsR0FBRztFQUN0QyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVM7SUFDckI7RUFDRjtFQUVBLE1BQU0sU0FBa0QsT0FBTyxNQUFNLENBQUM7RUFDdEUsSUFBSSxJQUFJO0VBQ1IsSUFBSSxZQUFZO0lBQ2QsTUFBTSxPQUFPLFdBQVcsS0FBSyxDQUFDLEtBQUssR0FBRyxDQUFDLENBQUMsSUFBTSxFQUFFLElBQUksSUFBSSxHQUFHLENBQ3pEO0lBR0YsS0FBSyxNQUFNLENBQUMsS0FBSyxJQUFJLElBQUksS0FBTTtNQUM3QixNQUFNLFFBQVEsT0FBTyxHQUFHLENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksR0FBRyxDQUFDLElBQUksTUFBTSxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQyxHQUM5RCxJQUFJLEtBQUssQ0FBQyxHQUFHLElBQUksTUFBTSxHQUFHLEtBQzFCO01BRUosSUFBSSxRQUFRLE9BQU8sT0FBTztRQUN4QixJQUFJLFdBQVc7UUFDZjtNQUNGO01BRUEsTUFBTSxDQUFDLElBQUksR0FBRztJQUNoQjtFQUNGO0VBRUEsT0FBTztJQUFFO0lBQU07SUFBUztJQUFRO0lBQUc7RUFBRTtBQUN2QztBQUVBLFNBQVMsWUFBWSxNQUFjO0VBQ2pDLE1BQU0sVUFBVSxPQUFPLEtBQUssQ0FBQyxLQUFLLEdBQUcsQ0FBQyxDQUFDLElBQU0sRUFBRSxJQUFJO0VBRW5ELE1BQU0sYUFBcUMsRUFBRTtFQUM3QyxLQUFLLE1BQU0sQ0FBQyxPQUFPLE9BQU8sSUFBSSxRQUFRLE9BQU8sR0FBSTtJQUMvQyxNQUFNLFlBQVksZUFBZSxPQUFPLElBQUksSUFBSTtJQUVoRCxJQUFJLFdBQVc7TUFDYixXQUFXLElBQUksQ0FBQztJQUNsQjtFQUNGO0VBRUEsT0FBTztBQUNUO0FBRUEsU0FBUyxZQUFZLElBQTBCO0VBQzdDLE9BQU8sR0FBRyxLQUFLLElBQUksQ0FBQyxDQUFDLEVBQUUsS0FBSyxPQUFPLEVBQUU7QUFDdkM7QUFFQSxTQUFTLFFBQ1AsSUFBWSxFQUNaLElBQTBCLEVBQzFCLEtBQWE7RUFFYixNQUFNLElBQUksZUFBZSxNQUFNO0VBRS9CLElBQUksQ0FBQyxHQUFHO0lBQ047RUFDRjtFQUVBLElBQUksSUFBSTtFQUVSLElBQUksS0FBSyxJQUFJLENBQUMsV0FBVyxPQUFPLEVBQUUsSUFBSSxDQUFDLFdBQVcsSUFBSTtJQUNwRCxLQUFLO0VBQ1AsT0FBTyxJQUFJLEtBQUssSUFBSSxLQUFLLEtBQUs7SUFDNUI7RUFDRjtFQUVBLElBQUksS0FBSyxPQUFPLENBQUMsV0FBVyxPQUFPLEVBQUUsT0FBTyxDQUFDLFdBQVcsSUFBSTtJQUMxRCxLQUFLO0VBQ1AsT0FBTyxJQUFJLEtBQUssT0FBTyxLQUFLLEtBQUs7SUFDL0I7RUFDRjtFQUVBLE1BQU0sT0FBTyxPQUFPLElBQUksQ0FBQyxLQUFLLE1BQU07RUFDcEMsSUFBSSxLQUFLLE1BQU0sRUFBRTtJQUNmLElBQ0UsS0FBSyxLQUFLLENBQUMsQ0FBQyxNQUNWLENBQUMsS0FBSyxNQUFNLENBQUMsSUFBSSxJQUFJLEVBQUUsRUFBRSxXQUFXLE9BQ2xDLENBQUMsRUFBRSxNQUFNLENBQUMsSUFBSSxJQUFJLEVBQUUsRUFBRSxXQUFXLEtBRXJDO01BQ0EsS0FBSztJQUNQLE9BQU87TUFDTDtJQUNGO0VBQ0Y7RUFFQSxPQUFPO0lBQ0wsR0FBRztJQUNILEdBQUcsS0FBSyxDQUFDO0lBQ1QsR0FBRyxLQUFLLENBQUM7SUFDVDtFQUNGO0FBQ0Y7QUFFQSxTQUFTLHFCQUNQLElBQVksRUFDWixRQUFnQyxFQUNoQyxLQUFhO0VBRWIsSUFBSSxXQUF3QjtJQUFFLEdBQUcsQ0FBQztJQUFHLEdBQUc7SUFBRyxHQUFHO0lBQUcsR0FBRztFQUFNO0VBRTFELEtBQUssTUFBTSxXQUFXLFNBQVU7SUFDOUIsTUFBTSxPQUFPLFFBQVEsTUFBTSxTQUFTO0lBRXBDLElBQ0UsUUFDQSxDQUFDLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUM3QixDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FDaEMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxHQUN6QztNQUNBLFdBQVc7SUFDYjtFQUNGO0VBRUEsT0FBTztBQUNUO0FBRUEsT0FBTyxTQUFTLG9CQUNkLE1BQXNCLEVBQ3RCLFFBQW1CO0VBRW5CLE1BQU0sVUFBVSxZQUFZLFdBQVcsWUFBWSxRQUFRLFVBQVU7RUFFckUsSUFBSSxDQUFDLFVBQVU7SUFDYixPQUFPLFFBQ0osTUFBTSxDQUFDLFdBQ1AsSUFBSSxDQUFDLGNBQ0wsR0FBRyxDQUFDO0VBQ1Q7RUFFQSxNQUFNLGFBQWEsU0FBUyxHQUFHLENBQUMsQ0FBQyxNQUFNO0lBQ3JDLE9BQU8scUJBQXFCLE1BQU0sU0FBUztFQUM3QztFQUVBLE9BQU8sV0FDSixNQUFNLENBQUMsV0FDUCxJQUFJLENBQUMsY0FDTCxHQUFHLENBQUMsQ0FBQyxXQUFhLFFBQVEsQ0FBQyxXQUFXLE9BQU8sQ0FBQyxVQUFVO0FBQzdEIn0=
// denoCacheMetadata=17179346167065140032,565858701834249800