// Ported from js-yaml v3.13.1:
// https://github.com/nodeca/js-yaml/commit/665aadda42349dcae869f12040d9b10ef18d12da
// Copyright 2011-2015 by Vitaly Puzrin. All rights reserved. MIT license.
// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
const YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + // [1] year
"-([0-9][0-9])" + // [2] month
"-([0-9][0-9])$");
const YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + // [1] year
"-([0-9][0-9]?)" + // [2] month
"-([0-9][0-9]?)" + // [3] day
"(?:[Tt]|[ \\t]+)" + // ...
"([0-9][0-9]?)" + // [4] hour
":([0-9][0-9])" + // [5] minute
":([0-9][0-9])" + // [6] second
"(?:\\.([0-9]*))?" + // [7] fraction
"(?:[ \\t]*(Z|([-+])([0-9][0-9]?)" + // [8] tz [9] tz_sign [10] tz_hour
"(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data) {
  if (data === null) return false;
  if (YAML_DATE_REGEXP.exec(data) !== null) return true;
  if (YAML_TIMESTAMP_REGEXP.exec(data) !== null) return true;
  return false;
}
function constructYamlTimestamp(data) {
  let match = YAML_DATE_REGEXP.exec(data);
  if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data);
  if (match === null) {
    throw new Error("Cannot construct YAML timestamp: date resolve error");
  }
  // match: [1] year [2] month [3] day
  const year = +match[1];
  const month = +match[2] - 1; // JS month starts with 0
  const day = +match[3];
  if (!match[4]) {
    // no hour
    return new Date(Date.UTC(year, month, day));
  }
  // match: [4] hour [5] minute [6] second [7] fraction
  const hour = +match[4];
  const minute = +match[5];
  const second = +match[6];
  let fraction = 0;
  if (match[7]) {
    let partFraction = match[7].slice(0, 3);
    while(partFraction.length < 3){
      // milli-seconds
      partFraction += "0";
    }
    fraction = +partFraction;
  }
  // match: [8] tz [9] tz_sign [10] tz_hour [11] tz_minute
  let delta = null;
  if (match[9] && match[10]) {
    const tzHour = +match[10];
    const tzMinute = +(match[11] || 0);
    delta = (tzHour * 60 + tzMinute) * 60000; // delta in milli-seconds
    if (match[9] === "-") delta = -delta;
  }
  const date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
  if (delta) date.setTime(date.getTime() - delta);
  return date;
}
function representYamlTimestamp(date) {
  return date.toISOString();
}
export const timestamp = {
  tag: "tag:yaml.org,2002:timestamp",
  construct: constructYamlTimestamp,
  predicate (object) {
    return object instanceof Date;
  },
  kind: "scalar",
  represent: representYamlTimestamp,
  resolve: resolveYamlTimestamp
};
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=14399834181700432540,3960532253225840786