// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Applies the given selector to all elements of the given collection and
 * returns the min value of all elements. If an empty array is provided the
 * function will return undefined.
 *
 * @template T The type of the elements in the array.
 *
 * @param array The array to find the minimum element in.
 * @param selector The function to get the value to compare from each element.
 *
 * @returns The smallest value of the given function or undefined if there are
 * no elements.
 *
 * @example Basic usage
 * ```ts
 * import { minOf } from "@std/collections/min-of";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const inventory = [
 *   { name: "mustard", count: 2 },
 *   { name: "soy", count: 4 },
 *   { name: "tomato", count: 32 },
 * ];
 *
 * const minCount = minOf(inventory, (item) => item.count);
 *
 * assertEquals(minCount, 2);
 * ```
 */ export function minOf(array, selector) {
  let minimumValue;
  for (const element of array){
    const currentValue = selector(element);
    if (minimumValue === undefined || currentValue < minimumValue) {
      minimumValue = currentValue;
      continue;
    }
    if (Number.isNaN(currentValue)) {
      return currentValue;
    }
  }
  return minimumValue;
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=14148414690038583298,7259626294177464054