// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Creates a record by associating each element of the input array with a key
 * generated by the selector function.
 *
 * If the selector produces the same key for multiple elements, the latest one
 * will be used (overriding the ones before it).
 *
 * @template T Type of the elements in the input array.
 *
 * @param array The array to transform.
 * @param selector The function to extract the key from each element.
 *
 * @returns A record with the keys produced by the selector and the elements as
 * values.
 *
 * @example Basic usage
 * ```ts
 * import { associateBy } from "@std/collections/associate-by";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const users = [
 *   { id: "a2e", userName: "Anna" },
 *   { id: "5f8", userName: "Arnold" },
 *   { id: "d2c", userName: "Kim" },
 * ];
 *
 * const usersById = associateBy(users, (user) => user.id);
 *
 * assertEquals(usersById, {
 *   "a2e": { id: "a2e", userName: "Anna" },
 *   "5f8": { id: "5f8", userName: "Arnold" },
 *   "d2c": { id: "d2c", userName: "Kim" },
 * });
 * ```
 */ export function associateBy(array, selector) {
  const result = {};
  for (const element of array){
    result[selector(element)] = element;
  }
  return result;
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=10904295497450535025,12244881342051555869