// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Applies the given transformer to all values in the given record and returns a
 * new record containing the resulting keys associated to the last value that
 * produced them.
 *
 * @template T The type of the values in the input record.
 * @template O The type of the values in the output record.
 * @template K The type of the keys in the input and output records.
 *
 * @param record The record to map values from.
 * @param transformer The function to transform each value.
 *
 * @returns A new record with all values transformed by the given transformer.
 *
 * @example Basic usage
 * ```ts
 * import { mapValues } from "@std/collections/map-values";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const usersById = {
 *   a5ec: { name: "Mischa" },
 *   de4f: { name: "Kim" },
 * };
 * const namesById = mapValues(usersById, (user) => user.name);
 *
 * assertEquals(
 *   namesById,
 *   {
 *     a5ec: "Mischa",
 *     de4f: "Kim",
 *   },
 * );
 * ```
 */ export function mapValues(record, transformer) {
  // deno-lint-ignore no-explicit-any
  const result = {};
  const entries = Object.entries(record);
  for (const [key, value] of entries){
    const mappedValue = transformer(value, key);
    result[key] = mappedValue;
  }
  return result;
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=14135774939509881799,6281358418452253287