// Ported from js-yaml v3.13.1:
// https://github.com/nodeca/js-yaml/commit/665aadda42349dcae869f12040d9b10ef18d12da
// Copyright 2011-2015 by Vitaly Puzrin. All rights reserved. MIT license.
// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { Type } from "../type.ts";
import { isNegativeZero } from "../_utils.ts";
function isHexCode(c) {
  return 0x30 <= /* 0 */ c && c <= 0x39 || 0x41 <= /* A */ c && c <= 0x46 || 0x61 <= /* a */ c && c <= 0x66;
}
function isOctCode(c) {
  return 0x30 <= /* 0 */ c && c <= 0x37 /* 7 */ ;
}
function isDecCode(c) {
  return 0x30 <= /* 0 */ c && c <= 0x39 /* 9 */ ;
}
function resolveYamlInteger(data) {
  const max = data.length;
  let index = 0;
  let hasDigits = false;
  if (!max) return false;
  let ch = data[index];
  // sign
  if (ch === "-" || ch === "+") {
    ch = data[++index];
  }
  if (ch === "0") {
    // 0
    if (index + 1 === max) return true;
    ch = data[++index];
    // base 2, base 8, base 16
    if (ch === "b") {
      // base 2
      index++;
      for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (ch !== "0" && ch !== "1") return false;
        hasDigits = true;
      }
      return hasDigits && ch !== "_";
    }
    if (ch === "x") {
      // base 16
      index++;
      for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (!isHexCode(data.charCodeAt(index))) return false;
        hasDigits = true;
      }
      return hasDigits && ch !== "_";
    }
    // base 8
    for(; index < max; index++){
      ch = data[index];
      if (ch === "_") continue;
      if (!isOctCode(data.charCodeAt(index))) return false;
      hasDigits = true;
    }
    return hasDigits && ch !== "_";
  }
  // base 10 (except 0) or base 60
  // value should not start with `_`;
  if (ch === "_") return false;
  for(; index < max; index++){
    ch = data[index];
    if (ch === "_") continue;
    if (ch === ":") break;
    if (!isDecCode(data.charCodeAt(index))) {
      return false;
    }
    hasDigits = true;
  }
  // Should have digits and should not end with `_`
  if (!hasDigits || ch === "_") return false;
  // if !base60 - done;
  if (ch !== ":") return true;
  // base60 almost not used, no needs to optimize
  return /^(:[0-5]?[0-9])+$/.test(data.slice(index));
}
function constructYamlInteger(data) {
  let value = data;
  const digits = [];
  if (value.indexOf("_") !== -1) {
    value = value.replace(/_/g, "");
  }
  let sign = 1;
  let ch = value[0];
  if (ch === "-" || ch === "+") {
    if (ch === "-") sign = -1;
    value = value.slice(1);
    ch = value[0];
  }
  if (value === "0") return 0;
  if (ch === "0") {
    if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
    if (value[1] === "x") return sign * parseInt(value, 16);
    return sign * parseInt(value, 8);
  }
  if (value.indexOf(":") !== -1) {
    value.split(":").forEach((v)=>{
      digits.unshift(parseInt(v, 10));
    });
    let valueInt = 0;
    let base = 1;
    digits.forEach((d)=>{
      valueInt += d * base;
      base *= 60;
    });
    return sign * valueInt;
  }
  return sign * parseInt(value, 10);
}
function isInteger(object) {
  return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !isNegativeZero(object);
}
export const int = new Type("tag:yaml.org,2002:int", {
  construct: constructYamlInteger,
  defaultStyle: "decimal",
  kind: "scalar",
  predicate: isInteger,
  represent: {
    binary (obj) {
      return obj >= 0 ? `0b${obj.toString(2)}` : `-0b${obj.toString(2).slice(1)}`;
    },
    octal (obj) {
      return obj >= 0 ? `0${obj.toString(8)}` : `-0${obj.toString(8).slice(1)}`;
    },
    decimal (obj) {
      return obj.toString(10);
    },
    hexadecimal (obj) {
      return obj >= 0 ? `0x${obj.toString(16).toUpperCase()}` : `-0x${obj.toString(16).toUpperCase().slice(1)}`;
    }
  },
  resolve: resolveYamlInteger,
  styleAliases: {
    binary: [
      2,
      "bin"
    ],
    decimal: [
      10,
      "dec"
    ],
    hexadecimal: [
      16,
      "hex"
    ],
    octal: [
      8,
      "oct"
    ]
  }
});
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=16580278242481890266,15751892193653192281