// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Applies the given selector to all elements of the provided collection and
 * returns the max value of all elements. If an empty array is provided the
 * function will return undefined.
 *
 * @template T The type of the elements in the array.
 *
 * @param array The array to find the maximum element in.
 * @param selector The function to get the value to compare from each element.
 *
 * @returns The largest value of the given function or undefined if there are no
 * elements.
 *
 * @example Basic usage
 * ```ts
 * import { maxOf } from "@std/collections/max-of";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const inventory = [
 *   { name: "mustard", count: 2 },
 *   { name: "soy", count: 4 },
 *   { name: "tomato", count: 32 },
 * ];
 *
 * const maxCount = maxOf(inventory, (item) => item.count);
 *
 * assertEquals(maxCount, 32);
 * ```
 */ export function maxOf(array, selector) {
  let maximumValue;
  for (const element of array){
    const currentValue = selector(element);
    if (maximumValue === undefined || currentValue > maximumValue) {
      maximumValue = currentValue;
      continue;
    }
    if (Number.isNaN(currentValue)) {
      return currentValue;
    }
  }
  return maximumValue;
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=7325808993312219248,13959262904583191960