// Ported from js-yaml v3.13.1:
// https://github.com/nodeca/js-yaml/commit/665aadda42349dcae869f12040d9b10ef18d12da
// Copyright 2011-2015 by Vitaly Puzrin. All rights reserved. MIT license.
// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { Type } from "../type.ts";
const YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + // [1] year
"-([0-9][0-9])" + // [2] month
"-([0-9][0-9])$");
const YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + // [1] year
"-([0-9][0-9]?)" + // [2] month
"-([0-9][0-9]?)" + // [3] day
"(?:[Tt]|[ \\t]+)" + // ...
"([0-9][0-9]?)" + // [4] hour
":([0-9][0-9])" + // [5] minute
":([0-9][0-9])" + // [6] second
"(?:\\.([0-9]*))?" + // [7] fraction
"(?:[ \\t]*(Z|([-+])([0-9][0-9]?)" + // [8] tz [9] tz_sign [10] tz_hour
"(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data) {
  if (data === null) return false;
  if (YAML_DATE_REGEXP.exec(data) !== null) return true;
  if (YAML_TIMESTAMP_REGEXP.exec(data) !== null) return true;
  return false;
}
function constructYamlTimestamp(data) {
  let match = YAML_DATE_REGEXP.exec(data);
  if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data);
  if (match === null) throw new Error("Date resolve error");
  // match: [1] year [2] month [3] day
  const year = +match[1];
  const month = +match[2] - 1; // JS month starts with 0
  const day = +match[3];
  if (!match[4]) {
    // no hour
    return new Date(Date.UTC(year, month, day));
  }
  // match: [4] hour [5] minute [6] second [7] fraction
  const hour = +match[4];
  const minute = +match[5];
  const second = +match[6];
  let fraction = 0;
  if (match[7]) {
    let partFraction = match[7].slice(0, 3);
    while(partFraction.length < 3){
      // milli-seconds
      partFraction += "0";
    }
    fraction = +partFraction;
  }
  // match: [8] tz [9] tz_sign [10] tz_hour [11] tz_minute
  let delta = null;
  if (match[9] && match[10]) {
    const tzHour = +match[10];
    const tzMinute = +(match[11] || 0);
    delta = (tzHour * 60 + tzMinute) * 60000; // delta in milli-seconds
    if (match[9] === "-") delta = -delta;
  }
  const date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
  if (delta) date.setTime(date.getTime() - delta);
  return date;
}
function representYamlTimestamp(date) {
  return date.toISOString();
}
export const timestamp = new Type("tag:yaml.org,2002:timestamp", {
  construct: constructYamlTimestamp,
  instanceOf: Date,
  kind: "scalar",
  represent: representYamlTimestamp,
  resolve: resolveYamlTimestamp
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vanNyLmlvL0BzdGQveWFtbC8wLjIyNC4zL190eXBlL3RpbWVzdGFtcC50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvLyBQb3J0ZWQgZnJvbSBqcy15YW1sIHYzLjEzLjE6XG4vLyBodHRwczovL2dpdGh1Yi5jb20vbm9kZWNhL2pzLXlhbWwvY29tbWl0LzY2NWFhZGRhNDIzNDlkY2FlODY5ZjEyMDQwZDliMTBlZjE4ZDEyZGFcbi8vIENvcHlyaWdodCAyMDExLTIwMTUgYnkgVml0YWx5IFB1enJpbi4gQWxsIHJpZ2h0cyByZXNlcnZlZC4gTUlUIGxpY2Vuc2UuXG4vLyBDb3B5cmlnaHQgMjAxOC0yMDI0IHRoZSBEZW5vIGF1dGhvcnMuIEFsbCByaWdodHMgcmVzZXJ2ZWQuIE1JVCBsaWNlbnNlLlxuXG5pbXBvcnQgeyBUeXBlIH0gZnJvbSBcIi4uL3R5cGUudHNcIjtcblxuY29uc3QgWUFNTF9EQVRFX1JFR0VYUCA9IG5ldyBSZWdFeHAoXG4gIFwiXihbMC05XVswLTldWzAtOV1bMC05XSlcIiArIC8vIFsxXSB5ZWFyXG4gICAgXCItKFswLTldWzAtOV0pXCIgKyAvLyBbMl0gbW9udGhcbiAgICBcIi0oWzAtOV1bMC05XSkkXCIsIC8vIFszXSBkYXlcbik7XG5cbmNvbnN0IFlBTUxfVElNRVNUQU1QX1JFR0VYUCA9IG5ldyBSZWdFeHAoXG4gIFwiXihbMC05XVswLTldWzAtOV1bMC05XSlcIiArIC8vIFsxXSB5ZWFyXG4gICAgXCItKFswLTldWzAtOV0/KVwiICsgLy8gWzJdIG1vbnRoXG4gICAgXCItKFswLTldWzAtOV0/KVwiICsgLy8gWzNdIGRheVxuICAgIFwiKD86W1R0XXxbIFxcXFx0XSspXCIgKyAvLyAuLi5cbiAgICBcIihbMC05XVswLTldPylcIiArIC8vIFs0XSBob3VyXG4gICAgXCI6KFswLTldWzAtOV0pXCIgKyAvLyBbNV0gbWludXRlXG4gICAgXCI6KFswLTldWzAtOV0pXCIgKyAvLyBbNl0gc2Vjb25kXG4gICAgXCIoPzpcXFxcLihbMC05XSopKT9cIiArIC8vIFs3XSBmcmFjdGlvblxuICAgIFwiKD86WyBcXFxcdF0qKFp8KFstK10pKFswLTldWzAtOV0/KVwiICsgLy8gWzhdIHR6IFs5XSB0el9zaWduIFsxMF0gdHpfaG91clxuICAgIFwiKD86OihbMC05XVswLTldKSk/KSk/JFwiLCAvLyBbMTFdIHR6X21pbnV0ZVxuKTtcblxuZnVuY3Rpb24gcmVzb2x2ZVlhbWxUaW1lc3RhbXAoZGF0YTogc3RyaW5nKTogYm9vbGVhbiB7XG4gIGlmIChkYXRhID09PSBudWxsKSByZXR1cm4gZmFsc2U7XG4gIGlmIChZQU1MX0RBVEVfUkVHRVhQLmV4ZWMoZGF0YSkgIT09IG51bGwpIHJldHVybiB0cnVlO1xuICBpZiAoWUFNTF9USU1FU1RBTVBfUkVHRVhQLmV4ZWMoZGF0YSkgIT09IG51bGwpIHJldHVybiB0cnVlO1xuICByZXR1cm4gZmFsc2U7XG59XG5cbmZ1bmN0aW9uIGNvbnN0cnVjdFlhbWxUaW1lc3RhbXAoZGF0YTogc3RyaW5nKTogRGF0ZSB7XG4gIGxldCBtYXRjaCA9IFlBTUxfREFURV9SRUdFWFAuZXhlYyhkYXRhKTtcbiAgaWYgKG1hdGNoID09PSBudWxsKSBtYXRjaCA9IFlBTUxfVElNRVNUQU1QX1JFR0VYUC5leGVjKGRhdGEpO1xuXG4gIGlmIChtYXRjaCA9PT0gbnVsbCkgdGhyb3cgbmV3IEVycm9yKFwiRGF0ZSByZXNvbHZlIGVycm9yXCIpO1xuXG4gIC8vIG1hdGNoOiBbMV0geWVhciBbMl0gbW9udGggWzNdIGRheVxuXG4gIGNvbnN0IHllYXIgPSArbWF0Y2hbMV0hO1xuICBjb25zdCBtb250aCA9ICttYXRjaFsyXSEgLSAxOyAvLyBKUyBtb250aCBzdGFydHMgd2l0aCAwXG4gIGNvbnN0IGRheSA9ICttYXRjaFszXSE7XG5cbiAgaWYgKCFtYXRjaFs0XSkge1xuICAgIC8vIG5vIGhvdXJcbiAgICByZXR1cm4gbmV3IERhdGUoRGF0ZS5VVEMoeWVhciwgbW9udGgsIGRheSkpO1xuICB9XG5cbiAgLy8gbWF0Y2g6IFs0XSBob3VyIFs1XSBtaW51dGUgWzZdIHNlY29uZCBbN10gZnJhY3Rpb25cblxuICBjb25zdCBob3VyID0gK21hdGNoWzRdO1xuICBjb25zdCBtaW51dGUgPSArbWF0Y2hbNV0hO1xuICBjb25zdCBzZWNvbmQgPSArbWF0Y2hbNl0hO1xuXG4gIGxldCBmcmFjdGlvbiA9IDA7XG4gIGlmIChtYXRjaFs3XSkge1xuICAgIGxldCBwYXJ0RnJhY3Rpb24gPSBtYXRjaFs3XS5zbGljZSgwLCAzKTtcbiAgICB3aGlsZSAocGFydEZyYWN0aW9uLmxlbmd0aCA8IDMpIHtcbiAgICAgIC8vIG1pbGxpLXNlY29uZHNcbiAgICAgIHBhcnRGcmFjdGlvbiArPSBcIjBcIjtcbiAgICB9XG4gICAgZnJhY3Rpb24gPSArcGFydEZyYWN0aW9uO1xuICB9XG5cbiAgLy8gbWF0Y2g6IFs4XSB0eiBbOV0gdHpfc2lnbiBbMTBdIHR6X2hvdXIgWzExXSB0el9taW51dGVcblxuICBsZXQgZGVsdGEgPSBudWxsO1xuICBpZiAobWF0Y2hbOV0gJiYgbWF0Y2hbMTBdKSB7XG4gICAgY29uc3QgdHpIb3VyID0gK21hdGNoWzEwXTtcbiAgICBjb25zdCB0ek1pbnV0ZSA9ICsobWF0Y2hbMTFdIHx8IDApO1xuICAgIGRlbHRhID0gKHR6SG91ciAqIDYwICsgdHpNaW51dGUpICogNjAwMDA7IC8vIGRlbHRhIGluIG1pbGxpLXNlY29uZHNcbiAgICBpZiAobWF0Y2hbOV0gPT09IFwiLVwiKSBkZWx0YSA9IC1kZWx0YTtcbiAgfVxuXG4gIGNvbnN0IGRhdGUgPSBuZXcgRGF0ZShcbiAgICBEYXRlLlVUQyh5ZWFyLCBtb250aCwgZGF5LCBob3VyLCBtaW51dGUsIHNlY29uZCwgZnJhY3Rpb24pLFxuICApO1xuXG4gIGlmIChkZWx0YSkgZGF0ZS5zZXRUaW1lKGRhdGUuZ2V0VGltZSgpIC0gZGVsdGEpO1xuXG4gIHJldHVybiBkYXRlO1xufVxuXG5mdW5jdGlvbiByZXByZXNlbnRZYW1sVGltZXN0YW1wKGRhdGU6IERhdGUpOiBzdHJpbmcge1xuICByZXR1cm4gZGF0ZS50b0lTT1N0cmluZygpO1xufVxuXG5leHBvcnQgY29uc3QgdGltZXN0YW1wID0gbmV3IFR5cGUoXCJ0YWc6eWFtbC5vcmcsMjAwMjp0aW1lc3RhbXBcIiwge1xuICBjb25zdHJ1Y3Q6IGNvbnN0cnVjdFlhbWxUaW1lc3RhbXAsXG4gIGluc3RhbmNlT2Y6IERhdGUsXG4gIGtpbmQ6IFwic2NhbGFyXCIsXG4gIHJlcHJlc2VudDogcmVwcmVzZW50WWFtbFRpbWVzdGFtcCxcbiAgcmVzb2x2ZTogcmVzb2x2ZVlhbWxUaW1lc3RhbXAsXG59KTtcbiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSwrQkFBK0I7QUFDL0Isb0ZBQW9GO0FBQ3BGLDBFQUEwRTtBQUMxRSwwRUFBMEU7QUFFMUUsU0FBUyxJQUFJLFFBQVEsYUFBYTtBQUVsQyxNQUFNLG1CQUFtQixJQUFJLE9BQzNCLDRCQUE0QixXQUFXO0FBQ3JDLGtCQUFrQixZQUFZO0FBQzlCO0FBR0osTUFBTSx3QkFBd0IsSUFBSSxPQUNoQyw0QkFBNEIsV0FBVztBQUNyQyxtQkFBbUIsWUFBWTtBQUMvQixtQkFBbUIsVUFBVTtBQUM3QixxQkFBcUIsTUFBTTtBQUMzQixrQkFBa0IsV0FBVztBQUM3QixrQkFBa0IsYUFBYTtBQUMvQixrQkFBa0IsYUFBYTtBQUMvQixxQkFBcUIsZUFBZTtBQUNwQyxxQ0FBcUMsa0NBQWtDO0FBQ3ZFO0FBR0osU0FBUyxxQkFBcUIsSUFBWTtFQUN4QyxJQUFJLFNBQVMsTUFBTSxPQUFPO0VBQzFCLElBQUksaUJBQWlCLElBQUksQ0FBQyxVQUFVLE1BQU0sT0FBTztFQUNqRCxJQUFJLHNCQUFzQixJQUFJLENBQUMsVUFBVSxNQUFNLE9BQU87RUFDdEQsT0FBTztBQUNUO0FBRUEsU0FBUyx1QkFBdUIsSUFBWTtFQUMxQyxJQUFJLFFBQVEsaUJBQWlCLElBQUksQ0FBQztFQUNsQyxJQUFJLFVBQVUsTUFBTSxRQUFRLHNCQUFzQixJQUFJLENBQUM7RUFFdkQsSUFBSSxVQUFVLE1BQU0sTUFBTSxJQUFJLE1BQU07RUFFcEMsb0NBQW9DO0VBRXBDLE1BQU0sT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO0VBQ3RCLE1BQU0sUUFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFLEdBQUksR0FBRyx5QkFBeUI7RUFDdkQsTUFBTSxNQUFNLENBQUMsS0FBSyxDQUFDLEVBQUU7RUFFckIsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLEVBQUU7SUFDYixVQUFVO0lBQ1YsT0FBTyxJQUFJLEtBQUssS0FBSyxHQUFHLENBQUMsTUFBTSxPQUFPO0VBQ3hDO0VBRUEscURBQXFEO0VBRXJELE1BQU0sT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO0VBQ3RCLE1BQU0sU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFO0VBQ3hCLE1BQU0sU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFO0VBRXhCLElBQUksV0FBVztFQUNmLElBQUksS0FBSyxDQUFDLEVBQUUsRUFBRTtJQUNaLElBQUksZUFBZSxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxHQUFHO0lBQ3JDLE1BQU8sYUFBYSxNQUFNLEdBQUcsRUFBRztNQUM5QixnQkFBZ0I7TUFDaEIsZ0JBQWdCO0lBQ2xCO0lBQ0EsV0FBVyxDQUFDO0VBQ2Q7RUFFQSx3REFBd0Q7RUFFeEQsSUFBSSxRQUFRO0VBQ1osSUFBSSxLQUFLLENBQUMsRUFBRSxJQUFJLEtBQUssQ0FBQyxHQUFHLEVBQUU7SUFDekIsTUFBTSxTQUFTLENBQUMsS0FBSyxDQUFDLEdBQUc7SUFDekIsTUFBTSxXQUFXLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxJQUFJLENBQUM7SUFDakMsUUFBUSxDQUFDLFNBQVMsS0FBSyxRQUFRLElBQUksT0FBTyx5QkFBeUI7SUFDbkUsSUFBSSxLQUFLLENBQUMsRUFBRSxLQUFLLEtBQUssUUFBUSxDQUFDO0VBQ2pDO0VBRUEsTUFBTSxPQUFPLElBQUksS0FDZixLQUFLLEdBQUcsQ0FBQyxNQUFNLE9BQU8sS0FBSyxNQUFNLFFBQVEsUUFBUTtFQUduRCxJQUFJLE9BQU8sS0FBSyxPQUFPLENBQUMsS0FBSyxPQUFPLEtBQUs7RUFFekMsT0FBTztBQUNUO0FBRUEsU0FBUyx1QkFBdUIsSUFBVTtFQUN4QyxPQUFPLEtBQUssV0FBVztBQUN6QjtBQUVBLE9BQU8sTUFBTSxZQUFZLElBQUksS0FBSywrQkFBK0I7RUFDL0QsV0FBVztFQUNYLFlBQVk7RUFDWixNQUFNO0VBQ04sV0FBVztFQUNYLFNBQVM7QUFDWCxHQUFHIn0=
// denoCacheMetadata=782160669544089796,15102467919050622515