// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
// Bare keys may only contain ASCII letters,
// ASCII digits, underscores, and dashes (A-Za-z0-9_-).
function joinKeys(keys) {
  // Dotted keys are a sequence of bare or quoted keys joined with a dot.
  // This allows for grouping similar properties together:
  return keys.map((str)=>{
    return str.length === 0 || str.match(/[^A-Za-z0-9_-]/) ? JSON.stringify(str) : str;
  }).join(".");
}
class Dumper {
  maxPad = 0;
  srcObject;
  output = [];
  #arrayTypeCache = new Map();
  constructor(srcObjc){
    this.srcObject = srcObjc;
  }
  dump(fmtOptions = {}) {
    // deno-lint-ignore no-explicit-any
    this.output = this.#printObject(this.srcObject);
    this.output = this.#format(fmtOptions);
    return this.output;
  }
  #printObject(obj, keys = []) {
    const out = [];
    const props = Object.keys(obj);
    const inlineProps = [];
    const multilineProps = [];
    for (const prop of props){
      if (this.#isSimplySerializable(obj[prop])) {
        inlineProps.push(prop);
      } else {
        multilineProps.push(prop);
      }
    }
    const sortedProps = inlineProps.concat(multilineProps);
    for (const prop of sortedProps){
      const value = obj[prop];
      if (value instanceof Date) {
        out.push(this.#dateDeclaration([
          prop
        ], value));
      } else if (typeof value === "string" || value instanceof RegExp) {
        out.push(this.#strDeclaration([
          prop
        ], value.toString()));
      } else if (typeof value === "number") {
        out.push(this.#numberDeclaration([
          prop
        ], value));
      } else if (typeof value === "boolean") {
        out.push(this.#boolDeclaration([
          prop
        ], value));
      } else if (value instanceof Array) {
        const arrayType = this.#getTypeOfArray(value);
        if (arrayType === "ONLY_PRIMITIVE") {
          out.push(this.#arrayDeclaration([
            prop
          ], value));
        } else if (arrayType === "ONLY_OBJECT_EXCLUDING_ARRAY") {
          // array of objects
          for(let i = 0; i < value.length; i++){
            out.push("");
            out.push(this.#headerGroup([
              ...keys,
              prop
            ]));
            out.push(...this.#printObject(value[i], [
              ...keys,
              prop
            ]));
          }
        } else {
          // this is a complex array, use the inline format.
          const str = value.map((x)=>this.#printAsInlineValue(x)).join(",");
          out.push(`${this.#declaration([
            prop
          ])}[${str}]`);
        }
      } else if (typeof value === "object") {
        out.push("");
        out.push(this.#header([
          ...keys,
          prop
        ]));
        if (value) {
          const toParse = value;
          out.push(...this.#printObject(toParse, [
            ...keys,
            prop
          ]));
        }
      // out.push(...this._parse(value, `${path}${prop}.`));
      }
    }
    out.push("");
    return out;
  }
  #isPrimitive(value) {
    return value instanceof Date || value instanceof RegExp || [
      "string",
      "number",
      "boolean"
    ].includes(typeof value);
  }
  #getTypeOfArray(arr) {
    if (this.#arrayTypeCache.has(arr)) {
      return this.#arrayTypeCache.get(arr);
    }
    const type = this.#doGetTypeOfArray(arr);
    this.#arrayTypeCache.set(arr, type);
    return type;
  }
  #doGetTypeOfArray(arr) {
    if (!arr.length) {
      // any type should be fine
      return "ONLY_PRIMITIVE";
    }
    const onlyPrimitive = this.#isPrimitive(arr[0]);
    if (arr[0] instanceof Array) {
      return "MIXED";
    }
    for(let i = 1; i < arr.length; i++){
      if (onlyPrimitive !== this.#isPrimitive(arr[i]) || arr[i] instanceof Array) {
        return "MIXED";
      }
    }
    return onlyPrimitive ? "ONLY_PRIMITIVE" : "ONLY_OBJECT_EXCLUDING_ARRAY";
  }
  #printAsInlineValue(value) {
    if (value instanceof Date) {
      return `"${this.#printDate(value)}"`;
    } else if (typeof value === "string" || value instanceof RegExp) {
      return JSON.stringify(value.toString());
    } else if (typeof value === "number") {
      return value;
    } else if (typeof value === "boolean") {
      return value.toString();
    } else if (value instanceof Array) {
      const str = value.map((x)=>this.#printAsInlineValue(x)).join(",");
      return `[${str}]`;
    } else if (typeof value === "object") {
      if (!value) {
        throw new Error("should never reach");
      }
      const str = Object.keys(value).map((key)=>{
        return `${joinKeys([
          key
        ])} = ${// deno-lint-ignore no-explicit-any
        this.#printAsInlineValue(value[key])}`;
      }).join(",");
      return `{${str}}`;
    }
    throw new Error("should never reach");
  }
  #isSimplySerializable(value) {
    return typeof value === "string" || typeof value === "number" || typeof value === "boolean" || value instanceof RegExp || value instanceof Date || value instanceof Array && this.#getTypeOfArray(value) !== "ONLY_OBJECT_EXCLUDING_ARRAY";
  }
  #header(keys) {
    return `[${joinKeys(keys)}]`;
  }
  #headerGroup(keys) {
    return `[[${joinKeys(keys)}]]`;
  }
  #declaration(keys) {
    const title = joinKeys(keys);
    if (title.length > this.maxPad) {
      this.maxPad = title.length;
    }
    return `${title} = `;
  }
  #arrayDeclaration(keys, value) {
    return `${this.#declaration(keys)}${JSON.stringify(value)}`;
  }
  #strDeclaration(keys, value) {
    return `${this.#declaration(keys)}${JSON.stringify(value)}`;
  }
  #numberDeclaration(keys, value) {
    switch(value){
      case Infinity:
        return `${this.#declaration(keys)}inf`;
      case -Infinity:
        return `${this.#declaration(keys)}-inf`;
      default:
        return `${this.#declaration(keys)}${value}`;
    }
  }
  #boolDeclaration(keys, value) {
    return `${this.#declaration(keys)}${value}`;
  }
  #printDate(value) {
    function dtPad(v, lPad = 2) {
      return v.padStart(lPad, "0");
    }
    const m = dtPad((value.getUTCMonth() + 1).toString());
    const d = dtPad(value.getUTCDate().toString());
    const h = dtPad(value.getUTCHours().toString());
    const min = dtPad(value.getUTCMinutes().toString());
    const s = dtPad(value.getUTCSeconds().toString());
    const ms = dtPad(value.getUTCMilliseconds().toString(), 3);
    // formatted date
    const fData = `${value.getUTCFullYear()}-${m}-${d}T${h}:${min}:${s}.${ms}`;
    return fData;
  }
  #dateDeclaration(keys, value) {
    return `${this.#declaration(keys)}${this.#printDate(value)}`;
  }
  #format(options = {}) {
    const { keyAlignment = false } = options;
    const rDeclaration = /^(\".*\"|[^=]*)\s=/;
    const out = [];
    for(let i = 0; i < this.output.length; i++){
      const l = this.output[i];
      // we keep empty entry for array of objects
      if (l[0] === "[" && l[1] !== "[") {
        // non-empty object with only subobjects as properties
        if (this.output[i + 1] === "" && this.output[i + 2]?.slice(0, l.length) === l.slice(0, -1) + ".") {
          i += 1;
          continue;
        }
        out.push(l);
      } else {
        if (keyAlignment) {
          const m = rDeclaration.exec(l);
          if (m && m[1]) {
            out.push(l.replace(m[1], m[1].padEnd(this.maxPad)));
          } else {
            out.push(l);
          }
        } else {
          out.push(l);
        }
      }
    }
    // Cleaning multiple spaces
    const cleanedOutput = [];
    for(let i = 0; i < out.length; i++){
      const l = out[i];
      if (!(l === "" && out[i + 1] === "")) {
        cleanedOutput.push(l);
      }
    }
    return cleanedOutput;
  }
}
/**
 * Stringify dumps source object into TOML string and returns it.
 *
 * @example Stringify an object
 * ```ts
 * import { stringify } from "@std/toml/stringify";
 * import { assertEquals } from "@std/assert/assert-equals";
 *
 * const obj = {
 *   title: "TOML Example",
 *   owner: {
 *     name: "Bob",
 *     bio: "Bob is a cool guy",
 *  }
 * };
 * const tomlString = stringify(obj);
 * assertEquals(tomlString, `title = "TOML Example"\n\n[owner]\nname = "Bob"\nbio = "Bob is a cool guy"\n`);
 * ```
 * @param srcObj Source object
 * @param fmtOptions format options
 * @returns TOML string
 */ export function stringify(srcObj, fmtOptions) {
  return new Dumper(srcObj).dump(fmtOptions).join("\n");
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=14045529511438006512,17599903418401249440