// Copyright 2014-2021 Sindre Sorhus. All rights reserved. MIT license.
// Copyright 2021 Yoshiya Hinosawa. All rights reserved. MIT license.
// Copyright 2021 Giuseppe Eletto. All rights reserved. MIT license.
// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Convert bytes to a human-readable string: 1337 → 1.34 kB
 *
 * Based on {@link https://github.com/sindresorhus/pretty-bytes | pretty-bytes}.
 * A utility for displaying file sizes for humans.
 *
 * This module is browser compatible.
 *
 * @example
 * ```ts
 * import { format } from "https://deno.land/std@$STD_VERSION/fmt/bytes.ts";
 *
 * format(1337);
 * //=> '1.34 kB'
 *
 * format(100);
 * //=> '100 B'
 *
 * // Display with units of bits
 * format(1337, { bits: true });
 * //=> '1.34 kbit'
 *
 * // Display file size differences
 * format(42, { signed: true });
 * //=> '+42 B'
 *
 * // Localized output using German locale
 * format(1337, { locale: "de" });
 * //=> '1,34 kB'
 * ```
 */ export function format(num, options = {}) {
  if (!Number.isFinite(num)) {
    throw new TypeError(`Expected a finite number, got ${typeof num}: ${num}`);
  }
  const UNITS_FIRSTLETTER = (options.bits ? "b" : "B") + "kMGTPEZY";
  if (options.signed && num === 0) {
    return ` 0 ${UNITS_FIRSTLETTER[0]}`;
  }
  const prefix = num < 0 ? "-" : options.signed ? "+" : "";
  num = Math.abs(num);
  const localeOptions = getLocaleOptions(options);
  if (num < 1) {
    const numberString = toLocaleString(num, options.locale, localeOptions);
    return prefix + numberString + " " + UNITS_FIRSTLETTER[0];
  }
  const exponent = Math.min(Math.floor(options.binary ? Math.log(num) / Math.log(1024) : Math.log10(num) / 3), UNITS_FIRSTLETTER.length - 1);
  num /= Math.pow(options.binary ? 1024 : 1000, exponent);
  if (!localeOptions) {
    num = Number(num.toPrecision(3));
  }
  const numberString = toLocaleString(num, options.locale, localeOptions);
  let unit = UNITS_FIRSTLETTER[exponent];
  if (exponent > 0) {
    unit += options.binary ? "i" : "";
    unit += options.bits ? "bit" : "B";
  }
  return prefix + numberString + " " + unit;
}
function getLocaleOptions({ maximumFractionDigits, minimumFractionDigits }) {
  if (maximumFractionDigits || minimumFractionDigits) {
    return {
      maximumFractionDigits,
      minimumFractionDigits
    };
  }
}
/**
 * Formats the given number using `Number#toLocaleString`.
 * - If locale is a string, the value is expected to be a locale-key (for example: `de`).
 * - If locale is true, the system default locale is used for translation.
 * - If no value for locale is specified, the number is returned unmodified.
 */ function toLocaleString(num, locale, options) {
  if (typeof locale === "string" || Array.isArray(locale)) {
    return num.toLocaleString(locale, options);
  } else if (locale === true || options !== undefined) {
    return num.toLocaleString(undefined, options);
  }
  return num.toString();
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=12448364157582571058,9779890520369938828